%***********************************************************************%
%  SUPPLEMENTARY CODE FOR THE ARTICLE TITLED:
%  " Reconstruction and Flux Analysis of Coupling Between Metabolic
%    Pathways of Astrocytes and Neurons: Application to Cerebral Hypoxia"
%
% The code identifies steady-state metabolic fluxes of astrocytes and
%    neurons at resting state using LP and QP.
% Moma approach is used to calculate fluxes at hypoxic state using QP.
%
% MATLAB interface to CLP solver is used to solve optimization problems.
%    http://www.coin-or.org/Clp/index.html
%    http://control.ee.ethz.ch/~joloef/clp.php
%***********************************************************************%
clear all

cd text2mat
eval('text2mat_brain');
cd ..
[metnum rxnnum] = size(stoich_sparse);
% Required variables  from text2mat_brain.m execution are:
% stoich_sparse, metname, rxnname, lb, ub.
%-------------------------------------------------

extmet_sec = {'LAC_AEX' 'LAC_NEX' 'CO2_AEX' 'CO2_NEX' 'GLN_AEX' 'DOP_NEX' ...
               'NOREP_NEX' 'EP_NEX' 'NOREP_AEX' 'SERT_NEX' 'MELT_NEX' 'ACHLN_NEX' 'Lipid_NEX' ...
               'Lipid_AEX' 'ReducedGlutathione_NEX' 'ReducedGlutathione_AEX'};
for j = 1:length(extmet_sec) % Released external metabolites
    a      = strmatch(extmet_sec{j},metname,'exact');
    rxnnum = rxnnum + 1;
    stoich_sparse(a,rxnnum) = -1;
    lb(rxnnum) = 0; 
    ub(rxnnum) = inf;  
end

extmet_up = {'GLC_AEX' 'GLC_NEX' 'O2_AEX' 'O2_NEX' 'LEU_AEX' 'ILE_AEX'... 
             'VAL_AEX' 'TYR_NEX' 'TRP_NEX' 'LYS_NEX' 'PHE_NEX' 'NH3_AEX'...
             'Linoleate_AEX' 'Linolenate_AEX' 'Choline_NEX' 'Choline_AEX' 'Cystine_AEX' 'GLG_AEX'};
for j = 1:length(extmet_up) % Consumed external metabolites
    a      = strmatch(extmet_up{j},metname,'exact');
    rxnnum = rxnnum + 1;
    stoich_sparse(a,rxnnum) = -1;
    lb(rxnnum) = -inf; 
    ub(rxnnum) = 0;
end

allrxnname = [rxnname; extmet_sec'; extmet_up'];

A = stoich_sparse;

df= size(A,2) - rank(full(A)); %degrees of freedom of A matrix


% MEASUREMENTS as LB and UB CONSTRAINTS
%--------------------------------------
[measname, mlb, mub] = textread('measdata.txt', '%s %f %f', 'delimiter','\t','commentstyle','shell');

for h = 1:length(measname)
    lb(strmatch(measname{h},allrxnname,'exact')) = mlb(h);  
    ub(strmatch(measname{h},allrxnname,'exact')) = mub(h);  
end
Rmeas = mlb;
ub(strmatch('R14a',allrxnname,'exact'))   = -0.06*Rmeas(1) ; % Astrocytic PPP was constrained to 6% of glycolytic flux: 11.01.06. TC.
ub(strmatch('R50n',allrxnname,'exact'))   = -0.05*Rmeas(2);
ub(strmatch('LACntr',allrxnname,'exact')) = 0; % no LAC secrn from neurons at steady state.
ub(strmatch('GLGatr',allrxnname,'exact')) = 0; % no glycogen  breakdown at steady state.

% ======================
% LINPROG APPLICATION
% ======================
[m,n] = size(A);

F             = zeros(1,n); 
F([75 78 81]) = -1;

B = zeros(m,1);
 A(end+1,81) = 1; A(end,78) = -0.25; B(end + 1) = 0; % making GABA flux 25% of Glutamate-Glutamine cycle flux
[X,z,status] = clp([],F,[],[],A,B,lb,ub);
objfun       = F*X;

% ========================================================
%               QUADPOG APPLICATION
%  objfun value obtained above is used as a constraint,
%  and QuadProg is used to find a flux distribution with 
%  minimum Euclidean norm among possible alternate optima.
% =========================================================

H          = eye(n,n);     
s = strmatch('GLCatr',allrxnname,'exact');
H(s:n,s:n) = 0; % GLCatr and following rxns in the reaction list are measured reactions. Therefore, they are excluded in sum of squre minimization.

F        = zeros(1,n);
A(end+1,[75 78 81]) = 1; 
B(end+1) = -objfun; % LP objfun is used as additional constraint.

disp('Resting State Flux Distribution by maximizing GLU/GLN/GABA cycle fluxes ...')
disp(' with subsequent minimization of Euclidean norm of Fluxes:')
[Xq,z,status] = clp(H,F,[],[],A,B,lb,ub);
opt_output(Xq,A, allrxnname)

 %===============================
 % HYPOXIA SIMULATIONS WITH MOMA
 %===============================
A(end,:) = []; %Remove objfun constraint for MOMA
A(end,:) = []; %Remove GABA constraint for MOMA
Am = A;     lbm = lb;      ubm = ub; 
ubm(14) = inf; % PPP fluxes were not constrained in MOMA.
ubm(50) = inf;
c1 = strmatch('CO2_AEX', allrxnname);
c2 = strmatch('CO2_NEX', allrxnname);
ubm([c1 c2]) = [inf inf];  lbm([c1 c2]) = [0 0]; % no constraint on RQ in MOMA

[m,n]     = size(Am);
Bm        = zeros(m,1);
ubm(strmatch('LACntr',allrxnname,'exact')) = inf; % LAC secrn from NEUR- for cerebral hypoxia
ubm(strmatch('GLGatr',allrxnname,'exact')) = inf; % active glycogen  breakdown during Hypoxia.
lbm = lbm(:); ubm = ubm(:);
  
[measnameh, mlbh, mubh] = textread('measdata_HOX.txt', '%s %f %f', 'delimiter','\t','commentstyle','shell');
for h = 1:length(measnameh)
    lbm(strmatch(measnameh{h},allrxnname,'exact')) = mlbh(h);  
    ubm(strmatch(measnameh{h},allrxnname,'exact')) = mubh(h);  
end
% FOR cerebral hypoxia
% ---------------------
Rmeasm = Rmeas;     
for i = 1:100 
  lbm(strmatch('O2_AEX',allrxnname,'exact')) = (100-i)*Rmeasm(3)/100;     
  ubm(strmatch('O2_AEX',allrxnname,'exact')) = (100-i)*Rmeasm(3)/100;
  lbm(strmatch('O2_NEX',allrxnname,'exact')) = (100-i)*Rmeasm(4)/100; 
  ubm(strmatch('O2_NEX',allrxnname,'exact')) = (100-i)*Rmeasm(4)/100;
  xaxis(i) = lbm(strmatch('O2_AEX',allrxnname,'exact')) + lbm(strmatch('O2_NEX',allrxnname,'exact')); 
 %  xaxis(i) = lbm(strmatch('O2_AEX',allrxnname,'exact')); % for astrocytic hypoxia
    
% measured fluxes are not accounted in MOMA minimization:
s = strmatch('GLCatr',allrxnname,'exact');
       Hm = eye(n,n); Hm(s:end,s:end) = 0;
       Fm = -Xq(1:n);  Fm(s:end) = 0;
       [Xm,z,status(i)] = clp(Hm,Fm,[],[],Am,Bm,lbm,ubm);
        objfunm(i) = 0.5*sum(Xm(1:s-1).^2) + Fm'*Xm;
                         
   yaxis(i,:) = [Xm(75) Xm(78) Xm(37) Xm(73)  Xm(22) Xm(58) Xm(11) Xm(48) Xm(68)...
                 Xm(81) Xm(87) Xm(106) Xm(1) Xm(38) Xm(183) ];
end
swt    = sum(Fm.^2);         % Sum of squares of resting-state fluxes used in Fm.
eudist = (objfunm + swt/2); % Euclidean distance between resting-state soln and MOMA solution

subplot(4,4,1);     plot(-xaxis,yaxis(:,1),'k', 'LineWidth',2);    title('Glutamate N->A r_7_5')
subplot(4,4,2);     plot(-xaxis,yaxis(:,2),'k', 'LineWidth',2);    title('Glutamine A->N r_7_8')
subplot(4,4,3);     plot(-xaxis,yaxis(:,3),'k', 'LineWidth',2);    title('ATP (A) r_3_7')
subplot(4,4,4);     plot(-xaxis,yaxis(:,4),'k', 'LineWidth',2);    title('ATP (N) r_7_3')
subplot(4,4,5);     plot(-xaxis,yaxis(:,5),'k', 'LineWidth',2);    title('TCA Cycle (A) r_2_2')
subplot(4,4,6);     plot(-xaxis,yaxis(:,6),'k', 'LineWidth',2);    title('TCA Cycle (N) r_5_8')
subplot(4,4,7);     plot(-xaxis,yaxis(:,7),'k', 'LineWidth',2);    title('Lactate (A) r_1_1')
subplot(4,4,8);     plot(-xaxis,yaxis(:,8),'k', 'LineWidth',2);   title('Lactate (N) r_4_8')
subplot(4,4,9);     plot(-xaxis,yaxis(:,9),'k', 'LineWidth',2);    title('Malate Shuttle (N) r_6_8')
subplot(4,4,10);    plot(-xaxis,yaxis(:,10),'k', 'LineWidth',2);   title('GABA N->A r_8_1')
subplot(4,4,11);    plot(-xaxis,yaxis(:,11),'k', 'LineWidth',2);   title('Aspartate N->A r_8_7')
subplot(4,4,12);    plot(-xaxis,yaxis(:,12),'k', 'LineWidth',2);   title('Leucine N->A r_1_0_6')
subplot(4,4,13);    plot(-xaxis,yaxis(:,13),'k', 'LineWidth',2);   title('Glucose(A) r_1')
subplot(4,4,14);    plot(-xaxis,yaxis(:,14),'k', 'LineWidth',2);   title('Glucose(N) r_3_8')
subplot(4,4,15);    plot(-xaxis,yaxis(:,15),'k', 'LineWidth',2);  title('Glycogen r_1_8_3')
subplot(4,4,16);    plot(-xaxis,eudist,'k', 'LineWidth',2);        title('objective function value')

disp('Key anoxic fluxes ...')
opt_output(Xm,Am, allrxnname) % key fluxes for anoxic state...
